<?php
// guardianapi/admin_login.php — adds organisation_name via JOIN, error-only logging
declare(strict_types=1);

require __DIR__ . '/config.php';

// Error-only logging (best effort to local error.log)
@ini_set('log_errors', '1');
@ini_set('error_log', __DIR__ . '/error.log');

// ---- CORS / preflight ----
gp_set_cors_all();
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  http_response_code(204);
  exit;
}
header('Content-Type: application/json; charset=utf-8');

// ---- Accept JSON or form/query ----
$in = gp_json_input();
$username = trim($in['username'] ?? ($_POST['username'] ?? ($_GET['username'] ?? '')));
$password = trim($in['password'] ?? ($_POST['password'] ?? ($_GET['password'] ?? '')));

if ($username === '' || $password === '') {
  gp_respond([
    'status'             => 'error',
    'token'              => null,
    'organisation_id'    => null,
    'organisation_name'  => null,
    'message'            => 'Please provide username and password.'
  ]);
}

$conn = getDbConnection();

try {
  // ---- Fetch admin + organisation_name via LEFT JOIN ----
  $sql = "SELECT a.admin_id,
                 a.organisation_id,
                 a.username,
                 a.password_hash,
                 a.token,
                 a.is_active,
                 o.organisation_name
          FROM admin a
          LEFT JOIN organisations o ON o.organisation_id = a.organisation_id
          WHERE a.username = ?
          LIMIT 1";
  $stmt = $conn->prepare($sql);
  if (!$stmt) { throw new RuntimeException('Prepare failed: ' . $conn->error); }
  $stmt->bind_param('s', $username);
  $stmt->execute();
  $res  = $stmt->get_result();
  $row  = $res->fetch_assoc();
  $stmt->close();

  if (!$row) {
    gp_respond([
      'status'             => 'error',
      'token'              => null,
      'organisation_id'    => null,
      'organisation_name'  => null,
      'message'            => 'User not found.'
    ]);
  }

  if ((int)$row['is_active'] !== 1) {
    gp_respond([
      'status'             => 'error',
      'token'              => null,
      'organisation_id'    => (int)$row['organisation_id'],
      'organisation_name'  => (string)($row['organisation_name'] ?? ''),
      'message'            => 'Account is disabled. Contact support.'
    ]);
  }

  // ---- Verify bcrypt password ----
  if (!password_verify($password, $row['password_hash'])) {
    gp_respond([
      'status'             => 'error',
      'token'              => null,
      'organisation_id'    => (int)$row['organisation_id'],
      'organisation_name'  => (string)($row['organisation_name'] ?? ''),
      'message'            => 'Wrong password.'
    ]);
  }

  // ---- Generate & save fresh token (32-hex) ----
  $newToken = bin2hex(random_bytes(16));
  $upd = $conn->prepare("UPDATE admin SET token = ? WHERE admin_id = ?");
  if (!$upd) { throw new RuntimeException('Prepare failed (update): ' . $conn->error); }
  $upd->bind_param('si', $newToken, $row['admin_id']);
  $upd->execute();
  $upd->close();

  gp_respond([
    'status'             => 'success',
    'token'              => $newToken,
    'organisation_id'    => (int)$row['organisation_id'],
    'organisation_name'  => (string)($row['organisation_name'] ?? ''),
    'message'            => 'Login successful.'
  ]);
} catch (Throwable $e) {
  // Errors only → error.log
  error_log('admin_login exception: ' . $e->getMessage());
  gp_respond([
    'status'             => 'error',
    'token'              => null,
    'organisation_id'    => null,
    'organisation_name'  => null,
    'message'            => 'Unexpected server error.'
  ], 500);
}
