<?php
/**
 * guardianapi/visitor_update_status.php
 * - Exact behavior of original: UPDATE visitor_records SET status=? WHERE id=?
 * - No assumptions (no updated_at, no extra columns)
 * - Accepts JSON body or form POST (website friendly)
 * - CORS + OPTIONS
 * - Errors logged to ./error.log
 */

declare(strict_types=1);

header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

require_once __DIR__ . '/config.php';

ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/error.log'); // write errors here

function vus_out(array $arr, int $code = 200): void {
  http_response_code($code);
  echo json_encode($arr);
  exit;
}

function vus_input(): array {
  $ct = strtolower(strtok((string)($_SERVER['CONTENT_TYPE'] ?? $_SERVER['HTTP_CONTENT_TYPE'] ?? ''), ';'));
  if ($ct === 'application/json' || $ct === 'text/json') {
    $raw = file_get_contents('php://input');
    $j = $raw !== false && $raw !== '' ? json_decode($raw, true) : null;
    if (is_array($j)) return $j;
  }
  // fallback to regular form POST (website)
  return $_POST ?? [];
}

try {
  $in = vus_input();

  if (!isset($in['visitor_id']) || !isset($in['status'])) {
    vus_out(["status" => "error", "message" => "Missing fields"], 400);
  }

  $visitorId = (int)$in['visitor_id'];
  $status    = (string)$in['status'];

  $conn = getDbConnection();
  $conn->set_charset('utf8mb4');

  $stmt = $conn->prepare("UPDATE visitor_records SET status = ? WHERE id = ?");
  if (!$stmt) {
    error_log("Prepare failed (update): " . $conn->error);
    vus_out(["status" => "error", "message" => "Update failed"], 500);
  }

  $stmt->bind_param("si", $status, $visitorId);

  if ($stmt->execute()) {
    // mirror original success payload
    vus_out(["status" => "success", "message" => "Visitor status updated"], 200);
  } else {
    error_log("Execute failed (update): " . $stmt->error);
    vus_out(["status" => "error", "message" => "Update failed"], 500);
  }

} catch (Throwable $e) {
  error_log("visitor_update_status fatal: " . $e->getMessage());
  vus_out(["status" => "error", "message" => "Update failed"], 500);
}
