<?php
// guardianapi/sms_send.php
// CORS-fixed, drop-in compatible with your current logic and schema.
declare(strict_types=1);

/* ---------------------- CORS ---------------------- */
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
$allowedOrigins = [
  'https://guestpassvms.com',
  // add more allowed web origins if needed:
  // 'https://www.guestpassvms.com',
  // 'http://localhost:5173',
  // 'http://127.0.0.1:5173',
];

if (in_array($origin, $allowedOrigins, true)) {
  header('Access-Control-Allow-Origin: ' . $origin);
} else {
  // (Optional) fallback to wildcard if you prefer — comment out if you want strict allowlist only
  header('Access-Control-Allow-Origin: *');
}
header('Vary: Origin');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Access-Control-Max-Age: 86400'); // cache preflight for a day

// Preflight must return early WITHOUT auth
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  http_response_code(204);
  exit;
}

header('Content-Type: application/json; charset=UTF-8');

/* ---------------------- Auth ---------------------- */
// Keep your existing auth requirement for non-preflight calls
if (!isset($GLOBALS['AUTH'])) {
  http_response_code(401);
  echo json_encode(['status'=>'error','message'=>'Unauthorized']);
  exit;
}
$auth = $GLOBALS['AUTH'];

require_once __DIR__ . '/config.php';
$conn = getDbConnection();

/* ---------------------- Input ---------------------- */
$input   = json_decode(file_get_contents('php://input'), true);
$orgId   = (int)($input['organisation_id'] ?? ($auth['organisation_id'] ?? 0));
$houseId = isset($input['house_id']) ? (int)$input['house_id'] : null;
$mobile  = preg_replace('/\s+/', '', (string)($input['mobile'] ?? ''));
$message = trim((string)($input['message'] ?? ''));

// Prefer TEXTSMS_* if provided; fall back to existing SMS_* constants.
$apiKey   = defined('TEXTSMS_APIKEY')   ? TEXTSMS_APIKEY   : (defined('SMS_APIKEY')   ? SMS_APIKEY   : '');
$partner  = defined('TEXTSMS_PARTNER')  ? TEXTSMS_PARTNER  : (defined('SMS_PARTNER')  ? SMS_PARTNER  : '');
$endpoint = defined('TEXTSMS_URL')      ? rtrim(TEXTSMS_URL,'/').'/' : 'https://sms.textsms.co.ke/api/services/sendotp/';
$short    = defined('SMS_SHORTCODE')    ? SMS_SHORTCODE    : (defined('TEXTSMS_SHORTCODE') ? TEXTSMS_SHORTCODE : 'GuestPass');
$provider = defined('SMS_PROVIDER')     ? SMS_PROVIDER     : 'TEXTSMS';

if ($orgId <= 0 || $mobile === '' || $message === '') {
  http_response_code(400);
  echo json_encode(['status'=>'error','message'=>'Missing organisation_id, mobile or message']);
  exit;
}

/* ---------------------- Quotas (unchanged) ---------------------- */
function quota_rows(mysqli $c, int $orgId, ?int $houseId): array {
  try {
    $sql="SELECT organisation_id, house_id, period, limit_count, window_start, window_end, is_active
            FROM sms_quotas
           WHERE is_active=1 AND organisation_id=? AND (house_id IS NULL OR house_id=?)";
    $st=$c->prepare($sql); if(!$st) return [];
    $hid=$houseId ?? 0; $st->bind_param("ii",$orgId,$hid);
    $st->execute(); $res=$st->get_result();
    $rows=[]; while($r=$res->fetch_assoc()) $rows[]=$r;
    $st->close(); return $rows;
  } catch(Throwable $e){ return []; }
}
function within_window(?string $start, ?string $end): bool {
  if (!$start && !$end) return true; $now=date('H:i:s');
  if ($start && $end) return ($now>=$start && $now<=$end);
  if ($start) return ($now>=$start); if ($end) return ($now<=$end); return true;
}
function count_used(mysqli $c, int $orgId, ?int $houseId, string $period): int {
  if ($period==='daily') {
    $sql="SELECT COUNT(*) n FROM sms_messages
          WHERE organisation_id=? AND (? IS NULL OR house_id=?)
            AND ((sent_at IS NOT NULL AND DATE(sent_at)=CURDATE())
                 OR (sent_at IS NULL AND DATE(NOW())=CURDATE()))
            AND status IN ('queued','sent')";
  } else {
    $sql="SELECT COUNT(*) n FROM sms_messages
          WHERE organisation_id=? AND (? IS NULL OR house_id=?)
            AND ((sent_at IS NOT NULL AND DATE_FORMAT(sent_at,'%Y-%m')=DATE_FORMAT(CURDATE(),'%Y-%m'))
                 OR (sent_at IS NULL AND DATE_FORMAT(NOW(),'%Y-%m')=DATE_FORMAT(CURDATE(),'%Y-%m')))
            AND status IN ('queued','sent')";
  }
  $st=$c->prepare($sql); $st->bind_param("iii",$orgId,$houseId,$houseId);
  $st->execute(); $res=$st->get_result(); $n=(int)($res->fetch_assoc()['n']??0); $st->close(); return $n;
}
function check_throttle(mysqli $c, int $orgId, ?int $houseId): ?array {
  $rows=quota_rows($c,$orgId,$houseId); if(!$rows) return null;
  $viol=[]; foreach($rows as $q){
    if(!within_window($q['window_start'],$q['window_end'])) continue;
    $used=count_used($c,$orgId,($q['house_id']!==null?(int)$q['house_id']:null),$q['period']);
    if($used>=(int)$q['limit_count']){
      $viol[]=['scope'=>$q['house_id']!==null?'house':'org','period'=>$q['period'],'limit'=>(int)$q['limit_count'],'used'=>$used];
    }
  }
  return $viol?['throttled'=>true,'violations'=>$viol]:null;
}

/* ---------------------- Queue row (same) ---------------------- */
$st=$conn->prepare("INSERT INTO sms_messages (organisation_id, house_id, recipient_mobile, message, status) VALUES (?,?,?,?, 'queued')");
$st->bind_param("iiss",$orgId,$houseId,$mobile,$message);
$st->execute(); $smsId=(int)$st->insert_id; $st->close();

/* ---------------------- Throttle check ---------------------- */
$th=check_throttle($conn,$orgId,$houseId);
if($th){
  $now=date('Y-m-d H:i:s');
  $st=$conn->prepare("UPDATE sms_messages SET status='throttled', sent_at=?, provider_response_desc='Quota exceeded' WHERE id=?");
  $st->bind_param("si",$now,$smsId); $st->execute(); $st->close();
  http_response_code(500);
  echo json_encode(['status'=>'error','message'=>'Quota exceeded','sms_id'=>$smsId,'provider_messageid'=>null]);
  exit;
}

/* ---------------------- Send via TEXTSMS endpoint ---------------------- */
$payload = [
  'apikey'    => $apiKey,
  'partnerID' => $partner,
  'mobile'    => $mobile,
  'message'   => $message,
  'shortcode' => $short,
];

$ch = curl_init($endpoint);
curl_setopt_array($ch, [
  CURLOPT_RETURNTRANSFER => true,
  CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
  CURLOPT_POST           => true,
  CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_UNICODE),
  CURLOPT_TIMEOUT        => 30,
]);
$respBody = curl_exec($ch);
$curlErr  = curl_error($ch);
$httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

/* ---------------------- Parse provider response ---------------------- */
$status = 'failed';
$msg    = 'Provider error';
$providerCode = $httpCode ?: 0;
$providerDesc = '';
$providerMsgId = null;

if ($curlErr) {
  $status='failed'; $msg='Transport error: '.$curlErr; $providerDesc=$msg;
} else {
  $json = json_decode((string)$respBody, true);

  if (is_array($json) && isset($json['responses'][0]) && is_array($json['responses'][0])) {
    $r = $json['responses'][0];
    // Handle both correct and common typo keys
    $providerCode = (int)($r['response-code'] ?? $r['respose-code'] ?? 0);
    $providerDesc = (string)($r['response-description'] ?? '');
    $providerMsgId = isset($r['messageid']) ? (string)$r['messageid'] : null;

    if ($providerCode === 200) {
      $status='sent'; $msg='SMS sent';
    } else {
      $status='failed'; $msg=$providerDesc ?: ('Provider error code '.$providerCode);
    }
  } else {
    // Fallback: treat clean 2xx as success to avoid false negatives
    if ($httpCode >= 200 && $httpCode < 300) {
      $status='sent'; $msg='SMS sent'; $providerDesc = (string)$respBody;
    } else {
      $status='failed'; $msg = (string)$respBody ?: ('HTTP '.$httpCode); $providerDesc=$msg;
    }
  }
}

/* ---------------------- Persist result ---------------------- */
$now = date('Y-m-d H:i:s');
$errText = ($status === 'failed' && $curlErr) ? $curlErr : null;

$st=$conn->prepare("UPDATE sms_messages
                       SET provider=?, provider_message_id=?, provider_response_code=?, provider_response_desc=?, status=?, sent_at=?, error=?
                     WHERE id=?");
$st->bind_param(
  "ssissssi",
  $provider,
  $providerMsgId,
  $providerCode,
  $providerDesc,
  $status,
  $now,
  $errText,
  $smsId
);
$st->execute(); $st->close();

/* ---------------------- Response (unchanged contract) ---------------------- */
http_response_code($status==='sent' ? 200 : 500);
echo json_encode([
  'status'  => $status==='sent' ? 'success' : 'error',
  'message' => $msg,
  'sms_id'  => $smsId,
  'provider_messageid' => $providerMsgId ?: null,
]);
