<?php
require_once __DIR__ . '/config.php';
header('Content-Type: application/json');

/** --------- safe file logger --------- */
$__ERROR_LOG_FILE = __DIR__ . '/error.log';
function logError($message, array $context = []) {
    global $__ERROR_LOG_FILE;
    foreach (['password','token','authorization','auth','secret','api_key'] as $k) {
        if (isset($context[$k])) $context[$k] = '[REDACTED]';
    }
    $line = sprintf(
        "[%s] ip=%s script=%s | %s | ctx=%s%s",
        date('c'),
        $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        basename(__FILE__),
        $message,
        json_encode($context, JSON_UNESCAPED_SLASHES),
        PHP_EOL
    );
    error_log($line, 3, $__ERROR_LOG_FILE);
}

try {
    $conn = getDbConnection();
    if (!$conn) {
        logError('DB connection failed');
        echo json_encode(["success"=>false,"status"=>"error","message"=>"Database connection failed"]);
        exit();
    }

    $raw  = file_get_contents('php://input');
    $data = json_decode($raw, true);
    if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
        logError('Invalid JSON', ['json_error'=>json_last_error_msg(), 'raw_len'=>strlen($raw)]);
        echo json_encode(["success"=>false,"status"=>"error","message"=>"Invalid JSON"]);
        exit();
    }

    // Required fields
    $missing = [];
    foreach (['real_name','username','password'] as $k) {
        if (!isset($data[$k]) || $data[$k] === '') $missing[] = $k;
    }

    // organisation_id may come as org_id
    $organisation_id = null;
    if (isset($data['organisation_id'])) $organisation_id = (int)$data['organisation_id'];
    elseif (isset($data['org_id']))     $organisation_id = (int)$data['org_id'];
    if (!$organisation_id) $missing[] = 'organisation_id';

    if ($missing) {
        logError('Missing required fields', ['missing'=>$missing]);
        echo json_encode(["success"=>false,"status"=>"error","message"=>"Invalid input"]);
        exit();
    }

    $real_name = trim((string)$data['real_name']);
    $username  = trim((string)$data['username']);
    $password  = (string)$data['password'];
    $house_id  = isset($data['house_id']) && $data['house_id'] !== '' ? (int)$data['house_id'] : null;

    // Optional is_active (default 1)
    $is_active = 1;
    if (isset($data['is_active'])) {
        $is_active = (int)$data['is_active'] ? 1 : 0;
    }

    $passwordHash = password_hash($password, PASSWORD_BCRYPT);
    if ($passwordHash === false) {
        logError('Password hashing failed');
        echo json_encode(["success"=>false,"status"=>"error","message"=>"Internal error"]);
        exit();
    }

    $token = bin2hex(random_bytes(32));

    // Optional: verify organisation exists
    if ($stmtCheck = $conn->prepare("SELECT org_id FROM organizations WHERE org_id = ?")) {
        $stmtCheck->bind_param("i", $organisation_id);
        if (!$stmtCheck->execute()) {
            logError('Execute failed (check org)', ['stmt_error'=>$stmtCheck->error,'organisation_id'=>$organisation_id]);
            echo json_encode(["success"=>false,"status"=>"error","message"=>"Database error"]);
            $stmtCheck->close();
            exit();
        }
        $res = $stmtCheck->get_result();
        if ($res && $res->num_rows === 0) {
            logError('Organization not found', ['organisation_id'=>$organisation_id]);
            echo json_encode(["success"=>false,"status"=>"error","message"=>"Invalid organization ID"]);
            $stmtCheck->close();
            exit();
        }
        $stmtCheck->close();
    }

    // Build insert (house_id nullable)
    if ($house_id === null) {
        $sql  = "INSERT INTO users (organisation_id, real_name, username, password, token, is_active)
                 VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            logError('Prepare failed (insert user, no house_id)', ['mysqli_error'=>$conn->error]);
            echo json_encode(["success"=>false,"status"=>"error","message"=>"Database error"]);
            exit();
        }
        $stmt->bind_param("issssi", $organisation_id, $real_name, $username, $passwordHash, $token, $is_active);
    } else {
        $sql  = "INSERT INTO users (organisation_id, real_name, username, password, token, is_active, house_id)
                 VALUES (?, ?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        if (!$stmt) {
            logError('Prepare failed (insert user, with house_id)', ['mysqli_error'=>$conn->error]);
            echo json_encode(["success"=>false,"status"=>"error","message"=>"Database error"]);
            exit();
        }
        $stmt->bind_param("issssii", $organisation_id, $real_name, $username, $passwordHash, $token, $is_active, $house_id);
    }

    if ($stmt->execute()) {
        echo json_encode([
            "success" => true,
            "status"  => "success",
            "message" => "User registered",
            "user_id" => (int)$conn->insert_id,
            "token"   => $token
        ]);
    } else {
        $err = $stmt->error;
        if (strpos($err, 'Duplicate') !== false) {
            logError('Duplicate entry', ['stmt_error'=>$err,'username'=>$username]);
            echo json_encode(["success"=>false,"status"=>"error","message"=>"Username already exists"]);
        } else {
            logError('Execute failed (insert user)', [
                'stmt_error'=>$err,'username'=>$username,
                'organisation_id'=>$organisation_id,'house_id'=>$house_id,'is_active'=>$is_active
            ]);
            echo json_encode(["success"=>false,"status"=>"error","message"=>"Failed to register user"]);
        }
    }

    $stmt->close();
    $conn->close();

} catch (Throwable $e) {
    logError('Unhandled exception', ['ex'=>$e->getMessage(),'trace'=>substr($e->getTraceAsString(),0,2000)]);
    echo json_encode(["success"=>false,"status"=>"error","message"=>"Internal server error"]);
}
