<?php
// guardianapi/org_update.php
declare(strict_types=1);
header('Content-Type: application/json; charset=UTF-8');

// GP config (provides getDbConnection() for GuestPass DB)
require_once __DIR__ . '/config.php';
// GA config (provides getGuardianDbConnection() for Guardian DB)
require_once __DIR__ . '/configguardian.php';

/* --------- helpers --------- */
if (!function_exists('gp_out')) {
  function gp_out(array $arr, int $code = 200): void {
    http_response_code($code);
    echo json_encode($arr);
    exit;
  }
}
if (!function_exists('gp_log')) {
  function gp_log(string $msg): void {
    @file_put_contents(
      __DIR__ . '/email_debug.log',
      date('Y-m-d H:i:s') . " - org_update: {$msg}\n",
      FILE_APPEND
    );
  }
}
function gp_conn_guestpass(): mysqli {
  if (function_exists('getDbConnection')) {
    $m = getDbConnection();
    if ($m instanceof mysqli) return $m;
  }
  throw new RuntimeException('GuestPass DB connection getDbConnection() not found.');
}
function gp_conn_guardian(): mysqli {
  if (function_exists('getGuardianDbConnection')) {
    $m = getGuardianDbConnection();
    if ($m instanceof mysqli) return $m;
  }
  throw new RuntimeException('Guardian DB connection getGuardianDbConnection() not found.');
}
function gp_table_exists(mysqli $conn, string $table): bool {
  $db = $conn->query('SELECT DATABASE()')->fetch_row()[0] ?? '';
  $dbEsc = $conn->real_escape_string($db);
  $tEsc  = $conn->real_escape_string($table);
  $sql = "SELECT 1 FROM INFORMATION_SCHEMA.TABLES
          WHERE TABLE_SCHEMA='{$dbEsc}' AND TABLE_NAME='{$tEsc}' LIMIT 1";
  $res = $conn->query($sql);
  $ok = $res && $res->num_rows > 0;
  if ($res) $res->free();
  return $ok;
}
function gp_col_exists(mysqli $conn, string $table, string $col): bool {
  $db = $conn->query('SELECT DATABASE()')->fetch_row()[0] ?? '';
  $dbEsc = $conn->real_escape_string($db);
  $tEsc  = $conn->real_escape_string($table);
  $cEsc  = $conn->real_escape_string($col);
  $sql = "SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA='{$dbEsc}' AND TABLE_NAME='{$tEsc}' AND COLUMN_NAME='{$cEsc}' LIMIT 1";
  $res = $conn->query($sql);
  $ok = $res && $res->num_rows > 0;
  if ($res) $res->free();
  return $ok;
}

/**
 * Detect org schema on a connection.
 * Returns:
 *   [
 *     'style' => 'US'|'UK',
 *     'table','id','name','email',
 *     'status','status_type' ('enum'|'tinyint'|null),
 *     'contact_name','contact_phone' (null if absent),
 *     'has_expiry' (bool), 'has_validated' (bool)
 *   ]
 */
function gp_detect_org_schema(mysqli $conn): array {
  // Guardian: organizations
  if (gp_table_exists($conn, 'organizations')
      && gp_col_exists($conn, 'organizations', 'org_id')
      && gp_col_exists($conn, 'organizations', 'org_name')) {

    $hasEmail  = gp_col_exists($conn, 'organizations', 'email');
    $hasStatus = gp_col_exists($conn, 'organizations', 'subscription_status');

    return [
      'style' => 'US',
      'table' => 'organizations',
      'id'    => 'org_id',
      'name'  => 'org_name',
      'email' => $hasEmail ? 'email' : null,
      'status'=> $hasStatus ? 'subscription_status' : null,
      'status_type' => $hasStatus ? 'enum' : null,
      'contact_name' => null,
      'contact_phone'=> null,
      'has_expiry'   => false,
      'has_validated'=> false,
    ];
  }

  // GuestPass: organisations
  if (gp_table_exists($conn, 'organisations')
      && gp_col_exists($conn, 'organisations', 'organisation_id')
      && gp_col_exists($conn, 'organisations', 'organisation_name')) {

    $hasEmail   = gp_col_exists($conn, 'organisations', 'contact_email');
    $hasName    = gp_col_exists($conn, 'organisations', 'contact_name');
    $hasPhone   = gp_col_exists($conn, 'organisations', 'contact_phone');
    $hasActive  = gp_col_exists($conn, 'organisations', 'is_active');
    $hasExpiry  = gp_col_exists($conn, 'organisations', 'expiry');
    $hasValid   = gp_col_exists($conn, 'organisations', 'validated');

    return [
      'style' => 'UK',
      'table' => 'organisations',
      'id'    => 'organisation_id',
      'name'  => 'organisation_name',
      'email' => $hasEmail ? 'contact_email' : null,
      'status'=> $hasActive ? 'is_active' : null,
      'status_type' => $hasActive ? 'tinyint' : null,
      'contact_name' => $hasName ? 'contact_name' : null,
      'contact_phone'=> $hasPhone ? 'contact_phone' : null,
      'has_expiry'   => $hasExpiry,
      'has_validated'=> $hasValid,
    ];
  }

  throw new RuntimeException('No org table detected on connection.');
}

function gp_row_exists(mysqli $conn, array $sch, int $id): bool {
  $stmt = $conn->prepare("SELECT 1 FROM `{$sch['table']}` WHERE `{$sch['id']}`=? LIMIT 1");
  $stmt->bind_param('i', $id);
  $stmt->execute();
  $stmt->store_result();
  $ok = $stmt->num_rows > 0;
  $stmt->close();
  return $ok;
}

/**
 * Build UPDATE for the detected schema using a neutral payload.
 * Neutral keys accepted:
 *   organisation_name | org_name
 *   contact_email | email
 *   contact_name, contact_phone (if present on schema)
 *   is_active (0/1) | subscription_status ('Active'|'Inactive')
 *   expiry, validated (if present on schema)
 */
function gp_build_update(mysqli $conn, array $sch, int $id, array $payload): void {
  $set = []; $types=''; $vals=[];

  // name
  $nameVal = $payload['organisation_name'] ?? $payload['org_name'] ?? null;
  if ($nameVal !== null) {
    $set[]  = "`{$sch['name']}`=?"; $types.='s'; $vals[]=(string)$nameVal;
  }

  // email (if column exists)
  if (!empty($sch['email'])) {
    if (array_key_exists('contact_email',$payload) || array_key_exists('email',$payload)) {
      $emailVal = $payload['contact_email'] ?? $payload['email'] ?? '';
      $set[]  = "`{$sch['email']}`=?"; $types.='s'; $vals[]=(string)$emailVal;
    }
  }

  // contact_name / contact_phone (UK only)
  if (!empty($sch['contact_name']) && array_key_exists('contact_name',$payload)) {
    $set[]="`{$sch['contact_name']}`=?"; $types.='s'; $vals[]=(string)$payload['contact_name'];
  }
  if (!empty($sch['contact_phone']) && array_key_exists('contact_phone',$payload)) {
    $set[]="`{$sch['contact_phone']}`=?"; $types.='s'; $vals[]=(string)$payload['contact_phone'];
  }

  // status mapping
  if (!empty($sch['status'])) {
    if ($sch['status_type']==='tinyint') {
      // UK side expects is_active (0/1). Accept is_active or subscription_status and convert.
      if (array_key_exists('is_active',$payload)) {
        $set[]="`{$sch['status']}`=?"; $types.='i'; $vals[]=(int)$payload['is_active']>0?1:0;
      } elseif (array_key_exists('subscription_status',$payload)) {
        $val = strtolower((string)$payload['subscription_status'])==='inactive' ? 0 : 1;
        $set[]="`{$sch['status']}`=?"; $types.='i'; $vals[]=$val;
      }
    } elseif ($sch['status_type']==='enum') {
      // US side expects subscription_status enum. Accept subscription_status or is_active and convert.
      if (array_key_exists('subscription_status',$payload)) {
        $val = strtolower((string)$payload['subscription_status'])==='inactive' ? 'Inactive' : 'Active';
        $set[]="`{$sch['status']}`=?"; $types.='s'; $vals[]=$val;
      } elseif (array_key_exists('is_active',$payload)) {
        $val = ((int)$payload['is_active']>0) ? 'Active' : 'Inactive';
        $set[]="`{$sch['status']}`=?"; $types.='s'; $vals[]=$val;
      }
    }
  }

  // expiry / validated if present on schema (UK side)
  if (!empty($sch['has_expiry']) && $sch['has_expiry'] && array_key_exists('expiry',$payload)) {
    $set[]="`expiry`=?"; $types.='s'; $vals[]=(string)$payload['expiry'];
  }
  if (!empty($sch['has_validated']) && $sch['has_validated'] && array_key_exists('validated',$payload)) {
    $set[]="`validated`=?"; $types.='i'; $vals[]=(int)$payload['validated'];
  }

  if (!$set) return; // nothing to update

  $sql = "UPDATE `{$sch['table']}` SET ".implode(',', $set)." WHERE `{$sch['id']}`=?";
  $types.='i'; $vals[]=$id;

  $stmt = $conn->prepare($sql);
  if (!$stmt) throw new RuntimeException("Prepare failed: ".$conn->error);
  $stmt->bind_param($types, ...$vals);
  if (!$stmt->execute()) {
    $err = $stmt->error; $stmt->close();
    throw new RuntimeException("Update failed: {$err}");
  }
  $stmt->close();
}

/* --------- auth (as you had it) --------- */
if (!isset($GLOBALS['AUTH'])) {
  http_response_code(401);
  echo json_encode(['status'=>'error','message'=>'Unauthorized']);
  exit;
}
$auth     = $GLOBALS['AUTH'];
$role     = (string)($auth['role'] ?? '');
$actorOrg = (int)($auth['organisation_id'] ?? -1);
$super    = ($role === 'admin') && (
              $actorOrg === 0 ||
              $actorOrg === 7 ||
              (defined('SUPERADMIN_ORG_ID') && $actorOrg === (int)SUPERADMIN_ORG_ID)
           );

$in    = json_decode(file_get_contents('php://input'), true) ?: [];
$orgId = (int)($in['organisation_id'] ?? 0);
if ($orgId <= 0) gp_out(['status'=>'error','message'=>'organisation_id required'], 400);
if (!$super && $orgId !== $actorOrg) gp_out(['status'=>'error','message'=>'Forbidden'], 403);

/* --------- core update (both DBs, no admin table) --------- */
try {
  $gp = gp_conn_guestpass();  // GuestPass DB
  $ga = gp_conn_guardian();   // Guardian DB

  $gpSch = gp_detect_org_schema($gp);
  $gaSch = gp_detect_org_schema($ga);

  $gpExists = gp_row_exists($gp, $gpSch, $orgId);
  $gaExists = gp_row_exists($ga, $gaSch, $orgId);

  if (!$gpExists || !$gaExists) {
    gp_out([
      'status'=>'error',
      'message'=>'Organisation not found in both systems',
      'missing'=>['guestpass'=>!$gpExists, 'guardian'=>!$gaExists]
    ], 404);
  }

  $gp->begin_transaction();
  $ga->begin_transaction();

  try {
    // Update on both sides with mapping
    gp_build_update($gp, $gpSch, $orgId, $in);
    gp_build_update($ga, $gaSch, $orgId, $in);

    $gp->commit();
    $ga->commit();

    gp_out(['status'=>'success','organisation_id'=>$orgId], 200);

  } catch (Throwable $e) {
    if ($gp->in_transaction) $gp->rollback();
    if ($ga->in_transaction) $ga->rollback();
    throw $e;
  }

} catch (Throwable $e) {
  gp_log('Fatal: '.$e->getMessage());
  gp_out(['status'=>'error','message'=>'Update failed: '.$e->getMessage()], 500);
}
