<?php
/**
 * guardianapi/house_edit.php
 * Edit or soft-delete an existing house.
 *
 * Identify the target by:
 *   - id
 *   - OR (organisation_id + house_number)
 *
 * Editable fields:
 *   - Rename house_number via either: new_house_number OR house_number_new
 *   - alert_email
 *   - alert_phone  (alias: alert_numbers)
 *   - is_active    (0/1, false/true, Inactive/Active)
 *   - active_until via: expires_at | expires_in_days | expires_in_months | expires_clear
 *   - status       ('Active'|'Deleted')  ← use to restore a deleted house
 *
 * Soft delete:
 *   - delete = 1  (or HTTP DELETE) → sets status='Deleted' (no hard delete)
 */

declare(strict_types=1);

require_once __DIR__ . '/config.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Access-Control-Allow-Methods: POST, PATCH, DELETE, OPTIONS');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }
if (!in_array($_SERVER['REQUEST_METHOD'], ['POST','PATCH','DELETE'], true)) {
  http_response_code(405);
  echo json_encode(['status'=>'error','message'=>'Method not allowed']);
  exit;
}

/* ---------------- Helpers ---------------- */
if (!function_exists('he_out')) {
  function he_out(array $arr, int $code = 200): void { http_response_code($code); echo json_encode($arr); exit; }
}
if (!function_exists('he_json')) {
  function he_json(): array { $j = json_decode(file_get_contents('php://input'), true); return is_array($j) ? $j : []; }
}
if (!function_exists('he_norm_bool')) {
  function he_norm_bool($v): int {
    if ($v === null) return 0;
    if (is_bool($v)) return $v ? 1 : 0;
    if (is_numeric($v)) return ((int)$v) > 0 ? 1 : 0;
    $s = strtolower(trim((string)$v));
    return in_array($s, ['1','true','yes','active'], true) ? 1 : 0;
  }
}
if (!function_exists('he_parse_datetime')) {
  function he_parse_datetime(?string $s): ?string {
    if ($s === null || trim($s) === '') return null;
    $s = trim($s);
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) return $s.' 23:59:59';
    if (preg_match('/^\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}(:\d{2})?$/', $s)) return strlen($s)===16 ? $s.':59' : $s;
    $ts = strtotime($s);
    return $ts === false ? null : date('Y-m-d H:i:s', $ts);
  }
}

/* ---------------- Optional auth (same pattern as others) ---------------- */
$auth     = $GLOBALS['AUTH'] ?? null;
$actorOrg = null;
$isSuper  = false;
if ($auth && is_array($auth)) {
  $role     = (string)($auth['role'] ?? '');
  $actorOrg = (int)($auth['organisation_id'] ?? -1);
  $isSuper  = ($role === 'admin') && (
    $actorOrg === 0 ||
    $actorOrg === 7 ||
    (defined('SUPERADMIN_ORG_ID') && $actorOrg === (int)SUPERADMIN_ORG_ID)
  );
}

/* ---------------- Input ---------------- */
$in = he_json();

$id              = isset($in['id']) ? (int)$in['id'] : 0;
$organisation_id = isset($in['organisation_id']) ? (int)$in['organisation_id'] : 0;
$house_number    = isset($in['house_number']) ? trim((string)$in['house_number']) : '';

if ($id <= 0 && (!($organisation_id > 0) || $house_number === '')) {
  he_out(['status'=>'error','message'=>'Provide id OR (organisation_id and house_number)'], 400);
}

/* Rename support (accept two parameter names) */
$new_house_number = null;
if (isset($in['new_house_number'])) {
  $new_house_number = trim((string)$in['new_house_number']);
} elseif (isset($in['house_number_new'])) {
  $new_house_number = trim((string)$in['house_number_new']);
}

/* Other editable fields */
$alert_email = isset($in['alert_email']) ? trim((string)$in['alert_email']) : null;
$alert_phone = isset($in['alert_phone']) ? trim((string)$in['alert_phone']) : null;
if ($alert_phone === null && isset($in['alert_numbers'])) $alert_phone = trim((string)$in['alert_numbers']);

$has_is_active = array_key_exists('is_active', $in) || array_key_exists('active', $in) || array_key_exists('status_active', $in);
$is_active     = $has_is_active ? he_norm_bool($in['is_active'] ?? $in['active'] ?? $in['status_active']) : null;

/* status: allow explicit set (Active/Deleted) */
$status = isset($in['status']) ? trim((string)$in['status']) : null;
if ($status !== null && !in_array($status, ['Active','Deleted'], true)) {
  he_out(['status'=>'error','message'=>"Invalid status. Use 'Active' or 'Deleted'."], 400);
}

/* expiry controls */
$expires_clear   = isset($in['expires_clear']) ? he_norm_bool($in['expires_clear']) === 1 : false;
$expires_at_raw  = isset($in['expires_at']) ? (string)$in['expires_at'] : null;
$expires_in_days = isset($in['expires_in_days']) ? max(0, (int)$in['expires_in_days']) : null;
$expires_in_mo   = isset($in['expires_in_months']) ? max(0, (int)$in['expires_in_months']) : null;

/* soft delete trigger */
$do_delete = isset($in['delete']) ? he_norm_bool($in['delete']) === 1 : (strtoupper($_SERVER['REQUEST_METHOD']) === 'DELETE');

/* ---------------- DB ---------------- */
$conn = getDbConnection();
$conn->set_charset('utf8mb4');
$conn->begin_transaction();

try {
  // Resolve target row
  if ($id > 0) {
    $stmt = $conn->prepare("SELECT id, organisation_id, house_number, alert_email, alert_phone, status, is_active, active_until FROM houses WHERE id=? LIMIT 1");
    $stmt->bind_param("i", $id);
  } else {
    $stmt = $conn->prepare("SELECT id, organisation_id, house_number, alert_email, alert_phone, status, is_active, active_until FROM houses WHERE organisation_id=? AND house_number=? LIMIT 1");
    $stmt->bind_param("is", $organisation_id, $house_number);
  }
  $stmt->execute();
  $res = $stmt->get_result();
  $row = $res ? $res->fetch_assoc() : null;
  $stmt->close();

  if (!$row) { $conn->rollback(); he_out(['status'=>'error','message'=>'House not found'], 404); }

  $hid   = (int)$row['id'];
  $orgId = (int)$row['organisation_id'];
  $hnum  = (string)$row['house_number'];

  // Org permission
  if ($auth && !$isSuper && $orgId !== (int)$actorOrg) {
    $conn->rollback();
    he_out(['status'=>'error','message'=>'Forbidden (org mismatch)'], 403);
  }

  // Soft delete
  if ($do_delete) {
    $stmt = $conn->prepare("UPDATE houses SET status='Deleted', updated_at=NOW() WHERE id=?");
    $stmt->bind_param("i", $hid);
    $stmt->execute();
    $stmt->close();

    $conn->commit();
    he_out([
      'status'  => 'success',
      'message' => 'House marked as Deleted',
      'data'    => ['id'=>$hid, 'organisation_id'=>$orgId, 'house_number'=>$hnum, 'status'=>'Deleted']
    ], 200);
  }

  // Build UPDATE
  $set   = [];
  $types = '';
  $vals  = [];

  // Rename house_number (collision check ignores rows with status='Deleted' so you can reuse old numbers)
  if ($new_house_number !== null && $new_house_number !== '' && $new_house_number !== $hnum) {
    if (strlen($new_house_number) > 100) $new_house_number = substr($new_house_number, 0, 100);

    $chk = $conn->prepare("SELECT id FROM houses WHERE organisation_id=? AND house_number=? AND status <> 'Deleted' AND id<>? LIMIT 1");
    $chk->bind_param("isi", $orgId, $new_house_number, $hid);
    $chk->execute();
    $dup = $chk->get_result()->fetch_assoc();
    $chk->close();

    if ($dup) { $conn->rollback(); he_out(['status'=>'error','message'=>'Another (non-deleted) house with that number already exists in this organisation'], 409); }

    $set[]  = "house_number=?";
    $types .= 's';
    $vals[] = $new_house_number;
    $hnum   = $new_house_number; // for response
  }

  if ($alert_email !== null) {
    if (strlen($alert_email) > 255) $alert_email = substr($alert_email, 0, 255);
    $set[]  = "alert_email=?";
    $types .= 's';
    $vals[] = $alert_email;
  }

  if ($alert_phone !== null) {
    if (strlen($alert_phone) > 20) $alert_phone = substr($alert_phone, 0, 20);
    $set[]  = "alert_phone=?";
    $types .= 's';
    $vals[] = $alert_phone;
  }

  if ($has_is_active) {
    $set[]  = "is_active=?";
    $types .= 'i';
    $vals[] = $is_active;
  }

  // explicit status set (e.g., restore from Deleted)
  if ($status !== null) {
    $set[]  = "status=?";
    $types .= 's';
    $vals[] = $status;
  }

  // expiry precedence
  if ($expires_clear) {
    $set[] = "active_until=NULL";
  } elseif ($expires_at_raw !== null) {
    $parsed = he_parse_datetime($expires_at_raw);
    if ($parsed === null) { $conn->rollback(); he_out(['status'=>'error','message'=>'Invalid expires_at format. Use YYYY-MM-DD or YYYY-MM-DD HH:MM[:SS]'], 400); }
    $set[]  = "active_until=?";
    $types .= 's';
    $vals[] = $parsed;
  } elseif ($expires_in_mo !== null) {
    $dt = new DateTime('now', new DateTimeZone(date_default_timezone_get()));
    if ($expires_in_mo > 0) $dt->modify('+'.$expires_in_mo.' months');
    $dt->setTime(23, 59, 59);
    $set[]  = "active_until=?";
    $types .= 's';
    $vals[] = $dt->format('Y-m-d H:i:s');
  } elseif ($expires_in_days !== null) {
    $dt = new DateTime('now', new DateTimeZone(date_default_timezone_get()));
    if ($expires_in_days > 0) $dt->modify('+'.$expires_in_days.' days');
    $dt->setTime(23, 59, 59);
    $set[]  = "active_until=?";
    $types .= 's';
    $vals[] = $dt->format('Y-m-d H:i:s');
  }

  if (empty($set)) { $conn->rollback(); he_out(['status'=>'error','message'=>'No editable fields provided'], 400); }

  $set[] = "updated_at=NOW()";
  $sql   = "UPDATE houses SET ".implode(',', $set)." WHERE id=?";
  $types .= 'i';
  $vals[]  = $hid;

  $stmt = $conn->prepare($sql);
  if (!$stmt) { $conn->rollback(); he_out(['status'=>'error','message'=>'Prepare failed: '.$conn->error], 500); }
  $stmt->bind_param($types, ...$vals);
  $stmt->execute();
  $stmt->close();

  // Read back
  $stmt = $conn->prepare("
    SELECT id, organisation_id, house_number, alert_email, alert_phone, status, is_active, active_until,
           CASE WHEN status='Active' AND is_active=1 AND (active_until IS NULL OR active_until >= NOW()) THEN 1 ELSE 0 END AS effective_active
    FROM houses WHERE id=? LIMIT 1
  ");
  $stmt->bind_param("i", $hid);
  $stmt->execute();
  $res  = $stmt->get_result();
  $data = $res ? $res->fetch_assoc() : null;
  $stmt->close();

  $conn->commit();

  he_out(['status'=>'success','message'=>'House updated','data'=>$data], 200);

} catch (Throwable $e) {
  $conn->rollback();
  he_out(['status'=>'error','message'=>'Edit failed: '.$e->getMessage()], 500);
}
