<?php
// host_visitor_get.php — strictly scoped to organisation + house
header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

// Error-only logging (best-effort to write to ./error.log)
$logDir = __DIR__;
$logFile = $logDir . '/error.log';
if (is_dir($logDir) && is_writable($logDir)) {
    @ini_set('log_errors', '1');
    @ini_set('error_log', $logFile);
}

require_once __DIR__ . '/config.php';
$conn = getDbConnection();
if (!$conn) {
    error_log("host_visitor_get: DB connection failed");
    http_response_code(500);
    echo json_encode(["status"=>"error","message"=>"Database connection failed"]);
    exit;
}

// Pagination
$page  = isset($_GET['page'])  ? max(1, (int)$_GET['page'])  : 1;
$limit = isset($_GET['limit']) ? max(1, (int)$_GET['limit']) : 20;
$offset = ($page - 1) * $limit;

// REQUIRED scope
$orgId   = isset($_GET['organisation_id']) ? (int)$_GET['organisation_id'] : 0;
$houseId = isset($_GET['house_id'])        ? (int)$_GET['house_id']        : 0;

if ($orgId === 0 || $houseId === 0) {
    http_response_code(400);
    echo json_encode(["status"=>"error","message"=>"Missing organisation_id or house_id"]);
    exit;
}

// Optional filters (kept simple to avoid over-filtering by default)
$conds = ["vr.organisation_id = ?", "vr.house_id = ?"];
$types = "ii";
$vals  = [ $orgId, $houseId ];

// By default exclude archived to keep host view clean; can override with include_archived=1
$includeArchived = isset($_GET['include_archived']) ? (int)$_GET['include_archived'] : 0;
if (!$includeArchived) {
    $conds[] = "vr.status <> 'archived'";
}

// Only unexited? (0/1)
$onlyUnexited = isset($_GET['only_unexited']) ? (int)$_GET['only_unexited'] : 0;
if ($onlyUnexited) {
    $conds[] = "vr.is_exited = 0";
}

// Exact filters
if (!empty($_GET['status'])) {
    $conds[] = "vr.status = ?";
    $types  .= "s";
    $vals[]  = (string)$_GET['status'];
}
if (!empty($_GET['phone_number'])) {
    $conds[] = "vr.phone_number = ?";
    $types  .= "s";
    $vals[]  = (string)$_GET['phone_number'];
}
if (!empty($_GET['id_plates'])) {
    $conds[] = "vr.id_plates = ?";
    $types  .= "s";
    $vals[]  = (string)$_GET['id_plates'];
}

// Fuzzy search across name/phone/plates
if (!empty($_GET['search'])) {
    $like = "%".(string)$_GET['search']."%";
    $conds[] = "(vr.name LIKE ? OR vr.phone_number LIKE ? OR vr.id_plates LIKE ?)";
    $types  .= "sss";
    $vals[]  = $like; $vals[] = $like; $vals[] = $like;
}

// Date range filter (created_at between date_from and date_to inclusive)
// Accepts YYYY-MM-DD or full datetime; we normalize to start/end of day if only a date is provided
$dateFrom = !empty($_GET['date_from']) ? (string)$_GET['date_from'] : null;
$dateTo   = !empty($_GET['date_to'])   ? (string)$_GET['date_to']   : null;
if ($dateFrom) {
    $ts = strtotime($dateFrom);
    if ($ts !== false) {
        $from = preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateFrom)
            ? date('Y-m-d 00:00:00', $ts)
            : date('Y-m-d H:i:s', $ts);
        $conds[] = "vr.created_at >= ?";
        $types  .= "s";
        $vals[]  = $from;
    }
}
if ($dateTo) {
    $ts = strtotime($dateTo);
    if ($ts !== false) {
        $to = preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateTo)
            ? date('Y-m-d 23:59:59', $ts)
            : date('Y-m-d H:i:s', $ts);
        $conds[] = "vr.created_at <= ?";
        $types  .= "s";
        $vals[]  = $to;
    }
}

$where = implode(" AND ", $conds);

// Main query
$sql = "
  SELECT
    vr.id, vr.house_id, h.house_number, vr.description, vr.name,
    vr.phone_number, vr.image_metadata, vr.id_plates, vr.admitted_by,
    vr.created_at, vr.otp_code, vr.is_exited, vr.status
  FROM visitor_records vr
  LEFT JOIN houses h ON vr.house_id = h.id
  WHERE $where
  ORDER BY vr.created_at DESC, vr.id DESC
  LIMIT ? OFFSET ?
";
$types .= "ii";
$vals[]  = $limit;
$vals[]  = $offset;

$stmt = $conn->prepare($sql);
if (!$stmt) {
    error_log("host_visitor_get: prepare failed: " . $conn->error);
    http_response_code(500);
    echo json_encode(["status"=>"error","message"=>"Failed to prepare statement"]);
    exit;
}

// bind_param with references
$bind = [$types];
foreach ($vals as $k => $v) { $bind[] = &$vals[$k]; }
call_user_func_array([$stmt, 'bind_param'], $bind);

if (!$stmt->execute()) {
    error_log("host_visitor_get: execute failed: " . $stmt->error);
    http_response_code(500);
    echo json_encode(["status"=>"error","message"=>"Query execution failed"]);
    $stmt->close();
    exit;
}

$res = $stmt->get_result();
$rows = [];
while ($row = $res->fetch_assoc()) { $rows[] = $row; }
$stmt->close();

// Optional: compute has_more cheaply
$has_more = (count($rows) === $limit);

echo json_encode([
  "status"         => "success",
  "page"           => $page,
  "limit"          => $limit,
  "total_fetched"  => count($rows),
  "has_more"       => $has_more,
  "visitor_records"=> $rows
], JSON_UNESCAPED_UNICODE);
exit;
