<?php
// clock_out.php
// Robust logging + safer error handling + clearer JSON responses

require __DIR__ . '/config.php';
date_default_timezone_set('Africa/Nairobi');

// ---------- Strict error logging ----------
ini_set('log_errors', '1');
ini_set('display_errors', '0');          // never echo PHP notices
ini_set('error_log', __DIR__ . '/api-error.log');
error_reporting(E_ALL);

// Simple structured logger
function apilog(string $msg, array $ctx = []): void {
    $line = '[clock_out] ' . $msg;
    if ($ctx) $line .= ' ' . json_encode($ctx, JSON_UNESCAPED_SLASHES);
    error_log($line);
}

// Convert PHP warnings/notices to exceptions so we can catch them
set_error_handler(function ($severity, $message, $file, $line) {
    if (!(error_reporting() & $severity)) return false;
    throw new ErrorException($message, 0, $severity, $file, $line);
});

// Catch unhandled throwables
set_exception_handler(function ($e) {
    apilog('UNCAUGHT', ['type'=>get_class($e), 'msg'=>$e->getMessage()]);
    if (!headers_sent()) {
        http_response_code(500);
        header('Content-Type: application/json; charset=utf-8');
    }
    echo json_encode(['success'=>false, 'message'=>'Server error', 'code'=>'UNCAUGHT']);
});

// Catch fatals (parse/oom/etc.) and ensure JSON
register_shutdown_function(function () {
    $err = error_get_last();
    if ($err && in_array($err['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR], true)) {
        apilog('FATAL', ['type'=>$err['type'], 'msg'=>$err['message'], 'file'=>$err['file'], 'line'=>$err['line']]);
        if (!headers_sent()) {
            http_response_code(500);
            header('Content-Type: application/json; charset=utf-8');
        }
        echo json_encode(['success'=>false, 'message'=>'Server error', 'code'=>'FATAL']);
    }
});

// ---------- CORS / headers ----------
header('Access-Control-Allow-Origin: *');
header('Vary: Origin');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Content-Type: application/json; charset=utf-8');

// Lightweight request id for correlating logs with client
$reqId = bin2hex(random_bytes(6));
header('X-Request-Id: ' . $reqId);

// Preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

// ---------- Helpers ----------
function pdo(): PDO {
    if (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) return $GLOBALS['pdo'];
    if (function_exists('getPdoConnection')) {
        $pdo = getPdoConnection();
        // Ensure exceptions for PDO
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return $pdo;
    }
    http_response_code(500);
    echo json_encode(['success'=>false, 'message'=>'DB not configured', 'code'=>'NO_DB']);
    apilog('NO_DB');
    exit;
}

function auth(PDO $pdo, string $token): ?array {
    // NEVER log full token
    $tTail = substr($token, -6);
    // Admin?
    $q = $pdo->prepare("SELECT id FROM admin_users WHERE token=? LIMIT 1");
    $q->execute([$token]);
    if ($q->fetchColumn()) {
        apilog('Auth ok (admin)', ['token_tail'=>$tTail]);
        return ['role'=>'admin', 'user_id'=>null];
    }
    // Guard?
    $q = $pdo->prepare("SELECT user_id FROM guardusers WHERE token=? LIMIT 1");
    $q->execute([$token]);
    $uid = $q->fetchColumn();
    if ($uid) {
        apilog('Auth ok (guard)', ['uid'=>(int)$uid, 'token_tail'=>$tTail]);
        return ['role'=>'user', 'user_id'=>(int)$uid];
    }
    apilog('Auth fail', ['token_tail'=>$tTail]);
    return null;
}

// ---------- Main ----------
try {
    $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
    apilog('REQ', ['id'=>$reqId, 'method'=>$method, 'uri'=>($_SERVER['REQUEST_URI'] ?? '')]);

    if ($method !== 'POST') {
        http_response_code(405);
        echo json_encode(['success'=>false, 'message'=>'Method not allowed', 'code'=>'METHOD']);
        apilog('METHOD_NOT_ALLOWED', ['got'=>$method]);
        exit;
    }

    // Read JSON safely
    $raw = file_get_contents('php://input') ?: '';
    $input = json_decode($raw, true);
    if ($input === null && json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['success'=>false, 'message'=>'Invalid JSON body', 'code'=>'BAD_JSON']);
        apilog('BAD_JSON', ['error'=>json_last_error_msg(), 'len'=>strlen($raw)]);
        exit;
    }
    if (!is_array($input)) $input = [];

    // Auth
    $authz = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $token = preg_replace('/^Bearer\s+/i', '', $authz);
    if (!$token) {
        http_response_code(401);
        echo json_encode(['success'=>false,'message'=>'Missing token','code'=>'NO_TOKEN']);
        apilog('NO_TOKEN');
        exit;
    }

    $pdo = pdo();
    $who = auth($pdo, $token);
    if (!$who) {
        http_response_code(401);
        echo json_encode(['success'=>false,'message'=>'Invalid token','code'=>'BAD_TOKEN']);
        exit;
    }

    // Inputs
    $user_id = (int)($input['user_id'] ?? 0);
    $org_id  = (int)($input['org_id']  ?? 0);
    $site_id = (int)($input['site_id'] ?? 0);
    $tag     = trim((string)($input['tag'] ?? ''));

    // Enforce guard scoping
    if ($who['role'] === 'user' && ($user_id === 0 || $user_id !== $who['user_id'])) {
        http_response_code(403);
        echo json_encode(['success'=>false,'message'=>'Forbidden (mismatched user)','code'=>'USER_SCOPE']);
        apilog('USER_SCOPE_FORBID', ['auth_user'=>$who['user_id'] ?? null, 'given_user'=>$user_id]);
        exit;
    }

    // Validate required
    $missing = [];
    if (!$user_id) $missing[] = 'user_id';
    if (!$org_id)  $missing[] = 'org_id';
    if (!$site_id) $missing[] = 'site_id';
    if ($tag === '') $missing[] = 'tag';
    if ($missing) {
        http_response_code(422);
        echo json_encode(['success'=>false,'message'=>'Missing required fields','missing'=>$missing,'code'=>'VALIDATION']);
        apilog('VALIDATION_FAIL', ['missing'=>$missing]);
        exit;
    }

    // Transactional update
    $pdo->beginTransaction();

    // Lock open session
    $sel = $pdo->prepare("
        SELECT id, site_id
        FROM book_sessions
        WHERE user_id = ? AND clock_out_time IS NULL
        ORDER BY id DESC
        LIMIT 1
        FOR UPDATE
    ");
    $sel->execute([$user_id]);
    $open = $sel->fetch(PDO::FETCH_ASSOC);

    if (!$open) {
        $pdo->rollBack();
        http_response_code(409);
        echo json_encode(['success'=>false,'message'=>'No open session.','code'=>'NO_OPEN']);
        apilog('NO_OPEN_SESSION', ['user_id'=>$user_id]);
        exit;
    }
    if ((int)$open['site_id'] !== $site_id) {
        $pdo->rollBack();
        http_response_code(409);
        echo json_encode(['success'=>false,'message'=>'Wrong site for clock-out.','code'=>'SITE_MISMATCH']);
        apilog('SITE_MISMATCH', ['expected'=>$open['site_id'], 'got'=>$site_id, 'user_id'=>$user_id]);
        exit;
    }

    $upd = $pdo->prepare("
        UPDATE book_sessions
        SET clock_out_tag = ?, clock_out_time = NOW()
        WHERE id = ?
    ");
    $upd->execute([$tag, (int)$open['id']]);

    $pdo->commit();

    apilog('CLOCK_OUT_OK', ['user_id'=>$user_id, 'session_id'=>(int)$open['id']]);
    echo json_encode(['success'=>true,'message'=>'Clock-out successful','data'=>['session_id'=>(int)$open['id']]]);
} catch (Throwable $e) {
    // Any runtime error
    if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    apilog('EXCEPTION', ['type'=>get_class($e), 'msg'=>$e->getMessage()]);
    http_response_code(500);
    echo json_encode(['success'=>false,'message'=>'Server error','code'=>'EXCEPTION']);
}
