<?php
// user_login.php — username+password only (multi-tenant safe)
// Table: users (user_id, organisation_id, real_name, username, password, token, house_id)

header("Content-Type: application/json; charset=utf-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

require_once __DIR__ . '/config.php'; // must provide getDbConnection(): mysqli

function read_input(): array {
    $raw = file_get_contents('php://input');
    if ($raw) {
        $json = json_decode($raw, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($json)) return $json;
    }
    return $_POST ?? [];
}
function json_error(string $msg, int $code = 400) {
    http_response_code($code);
    echo json_encode(["status" => "error", "message" => $msg], JSON_UNESCAPED_UNICODE);
    exit;
}
if (!function_exists('str_starts_with')) {
    function str_starts_with(string $h, string $n): bool {
        return $n !== '' && substr($h, 0, strlen($n)) === $n;
    }
}

try {
    $in   = read_input();
    $user = isset($in['username']) ? trim((string)$in['username']) : '';
    $pass = isset($in['password']) ? (string)$in['password'] : '';
    // Optional org for disambiguation
    $orgIdIn = isset($in['organisation_id']) && $in['organisation_id'] !== '' ? (int)$in['organisation_id'] : null;

    if ($user === '' || $pass === '') {
        json_error("Missing required fields: username, password.");
    }

    $db = getDbConnection(); // mysqli
    if (!$db) json_error("Database connection failed.", 500);
    $db->set_charset('utf8mb4');

    // 1) Fetch rows by username (optionally narrowed by organisation_id)
    if ($orgIdIn !== null) {
        $sql = "SELECT user_id, organisation_id, real_name, username, password, token, house_id
                FROM users WHERE username = ? AND organisation_id = ?";
        $st = $db->prepare($sql);
        if (!$st) json_error("Database error (prepare): ".$db->error, 500);
        $st->bind_param("si", $user, $orgIdIn);
    } else {
        $sql = "SELECT user_id, organisation_id, real_name, username, password, token, house_id
                FROM users WHERE username = ?";
        $st = $db->prepare($sql);
        if (!$st) json_error("Database error (prepare): ".$db->error, 500);
        $st->bind_param("s", $user);
    }

    $st->execute();
    $res = $st->get_result();
    if ($res->num_rows === 0) json_error("Invalid credentials.", 401);

    // 2) Verify password; collect matches
    $matches = [];
    while ($row = $res->fetch_assoc()) {
        $stored = (string)$row['password'];
        $isHash = str_starts_with($stored, '$2y$') || str_starts_with($stored, '$2a$') || str_starts_with($stored, '$argon2');
        $ok = $isHash ? password_verify($pass, $stored) : hash_equals($stored, $pass);
        if ($ok) $matches[] = $row;
    }
    $st->close();

    if (count($matches) === 0) json_error("Invalid credentials.", 401);
    if ($orgIdIn === null && count($matches) > 1) {
        // Username exists in multiple orgs — require orgId
        json_error("Multiple accounts found. Please include organisation_id.", 409);
    }

    $row = $matches[0];
    $uid = (int)$row['user_id'];

    // 3) REUSE existing token; generate only if missing
    $token = (string)($row['token'] ?? '');
    if ($token === '') {
        $token = bin2hex(random_bytes(32));
        $up = $db->prepare("UPDATE users SET token = ? WHERE user_id = ? LIMIT 1");
        if (!$up) json_error("Database error (prepare update): ".$db->error, 500);
        $up->bind_param("si", $token, $uid);
        $up->execute();
        $up->close();
    }

    // 4) Respond with minimal identity (Android will call userdatafetch.php next)
    $payload = [
        "status" => "success",
        "message" => "Login successful.",
        "token" => $token,

        // Both snake_case + camelCase for client compatibility
        "user_id" => $uid,
        "userId" => $uid,
        "organisation_id" => (int)$row['organisation_id'],
        "organisationId" => (int)$row['organisation_id'],
        "house_id" => isset($row['house_id']) ? (int)$row['house_id'] : null,
        "houseId" => isset($row['house_id']) ? (int)$row['house_id'] : null,
        "real_name" => (string)$row['real_name'],
        "realName" => (string)$row['real_name'],
        "username" => (string)$row['username'],
    ];

    echo json_encode($payload, JSON_UNESCAPED_UNICODE);
    $db->close();
} catch (Throwable $e) {
    json_error("Server error.", 500);
}
