<?php
/*  uploadVisitorImage.php
 *  --------------------------------------------------------------------
 *  Accept ONE multipart image + 4 text parts:
 *      - kind      = passport | id_front | id_back   (required)
 *      - user_id   = numeric                         (required)
 *      - house_id  = numeric                         (required)
 *      - image     = file field                      (required)
 *  Returns:  { status, message, imageUrl }
 *  ------------------------------------------------------------------ */

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Authorization, Content-Type");

require_once __DIR__ . '/config.php';            // ← your DB creds + UPLOAD_DIR
$conn = getDbConnection();
if (!$conn) {
    http_response_code(500);
    echo json_encode(["status"=>"error","message"=>"DB connection failed"]);
    exit();
}

/* ─── 1. HTTP & Auth checks ───────────────────────────────────────── */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(["status"=>"error","message"=>"POST only"]);
    exit();
}

$headers = getallheaders();
if (
    empty($headers['Authorization']) ||
    !preg_match('/Bearer\s(\S+)/', $headers['Authorization'], $m)
) {
    http_response_code(401);
    echo json_encode(["status"=>"error","message"=>"Missing token"]);
    exit();
}
$token = $m[1];

$stmt = $conn->prepare("SELECT user_id FROM users WHERE token = ?");
$stmt->bind_param("s", $token);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();
if (!$user) {
    http_response_code(401);
    echo json_encode(["status"=>"error","message"=>"Invalid token"]);
    exit();
}
$authUserId = (int)$user['user_id'];     // not used further, but verified

/* ─── 2. Validate multipart fields ────────────────────────────────── */
$kind     = $_POST['kind']    ?? '';
$userId   = $_POST['user_id'] ?? '';
$houseId  = $_POST['house_id']?? '';

$allowedKinds = ['passport','id_front','id_back'];
if (!in_array($kind,$allowedKinds, true)) {
    http_response_code(400);
    echo json_encode(["status"=>"error","message"=>"Invalid kind"]);
    exit();
}
if (!ctype_digit($userId) || !ctype_digit($houseId)) {
    http_response_code(400);
    echo json_encode(["status"=>"error","message"=>"user_id/house_id must be numeric"]);
    exit();
}
if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400);
    echo json_encode(["status"=>"error","message"=>"Image missing or corrupt"]);
    exit();
}
$file = $_FILES['image'];
$allowedTypes = ['image/jpeg','image/png'];
if (!in_array($file['type'],$allowedTypes, true)) {
    http_response_code(400);
    echo json_encode(["status"=>"error","message"=>"Only JPG/PNG allowed"]);
    exit();
}

/* ─── 3. Prepare upload directory ─────────────────────────────────── */
$base = $config['UPLOAD_DIR'] ?? (__DIR__.'/uploads/');
$houseDir = $base . 'house_' . $houseId . '/';
if (!is_dir($houseDir) && !mkdir($houseDir,0755,true)) {
    http_response_code(500);
    echo json_encode(["status"=>"error","message"=>"Failed to create dir"]);
    exit();
}

/* ─── 4. Move file ───────────────────────────────────────────────── */
$ext = pathinfo($file['name'], PATHINFO_EXTENSION);
$fname = uniqid($kind . '_') . '.' . $ext;
$path  = $houseDir . $fname;

if (!move_uploaded_file($file['tmp_name'], $path)) {
    http_response_code(500);
    echo json_encode(["status"=>"error","message"=>"Save failed"]);
    exit();
}

/* ─── 5. Record metadata ─────────────────────────────────────────── */
$stmt = $conn->prepare(
    "INSERT INTO visitorimages (user_id, house_id, kind, file_name) VALUES (?,?,?,?)"
);
$stmt->bind_param("iiss", $userId, $houseId, $kind, $fname);
$stmt->execute();

/* ─── 6. Build public URL & respond ──────────────────────────────── */
$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
$public  = $protocol . $_SERVER['HTTP_HOST'] . '/uploads/house_' . $houseId . '/' . $fname;

echo json_encode([
    "status"   => "success",
    "message"  => "Image uploaded",
    "imageUrl" => $public
]);
