<?php
// update_admin_password.php

require_once 'config.php';
header('Content-Type: application/json');

$conn = null;
try {
    $conn = getDbConnection();

    // parse JSON body
    $in    = json_decode(file_get_contents('php://input'), true);
    $token = trim($in['token']       ?? '');
    $old   = trim($in['oldPassword'] ?? '');
    $new   = trim($in['newPassword'] ?? '');

    if ($token === '' || $old === '' || $new === '') {
        http_response_code(400);
        echo json_encode([
            'status'  => 'error',
            'message' => 'token, oldPassword and newPassword are all required'
        ]);
        exit;
    }

    // lookup admin by token
    $stmt = $conn->prepare("
      SELECT admin_id, password_hash
      FROM admin
      WHERE token = ? AND is_active = 1
      LIMIT 1
    ");
    $stmt->bind_param('s', $token);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows === 0) {
        http_response_code(401);
        echo json_encode([
          'status'  => 'error',
          'message' => 'Invalid token or account inactive'
        ]);
        exit;
    }
    $stmt->bind_result($adminId, $currentHash);
    $stmt->fetch();
    $stmt->close();

    // verify old password
    if (!password_verify($old, $currentHash)) {
        http_response_code(403);
        echo json_encode([
          'status'  => 'error',
          'message' => 'Current password is incorrect'
        ]);
        exit;
    }

    // hash & update to new
    $newHash = password_hash($new, PASSWORD_DEFAULT);
    $up = $conn->prepare("
      UPDATE admin
      SET password_hash = ?
      WHERE admin_id = ?
    ");
    $up->bind_param('si', $newHash, $adminId);
    if (!$up->execute()) {
        throw new Exception($up->error);
    }
    $up->close();

    echo json_encode([
      'status'  => 'success',
      'message' => 'Password changed successfully'
    ]);

} catch (Exception $e) {
    if ($conn) $conn->rollback();
    http_response_code(500);
    error_log("update_admin_password error: ".$e->getMessage());
    echo json_encode([
      'status'  => 'error',
      'message' => 'Server error: '.$e->getMessage()
    ]);
} finally {
    if ($conn && $conn instanceof mysqli) $conn->close();
}
