<?php
// guardianapi/org_post.php
declare(strict_types=1);

header("Content-Type: application/json; charset=UTF-8");

// Load GuestPass DB config (has getDbConnection / DB_* )
require_once __DIR__ . '/config.php';
// Load Guardian Assist DB config (has getGuardianDbConnection / GA_DB_* )
require_once __DIR__ . '/configguardian.php';

/* ---------------------------------------------------------------------
   Helpers (prefixed to avoid naming clashes)
--------------------------------------------------------------------- */

if (!function_exists('gp_out')) {
  function gp_out(array $arr, int $code = 200): void {
    http_response_code($code);
    echo json_encode($arr);
    exit();
  }
}

if (!function_exists('gp_body_json')) {
  function gp_body_json(): array {
    $j = json_decode(file_get_contents('php://input'), true);
    return is_array($j) ? $j : [];
  }
}

if (!function_exists('gp_log')) {
  function gp_log(string $msg): void {
    @file_put_contents(
      __DIR__ . '/email_debug.log',
      date('Y-m-d H:i:s') . " - org_post: {$msg}\n",
      FILE_APPEND
    );
  }
}

/** Open mysqli connection for GuestPass DB (from config.php) */
function gp_conn_guestpass(): mysqli {
  if (function_exists('getDbConnection')) {
    $m = getDbConnection();
    if ($m instanceof mysqli) return $m;
  }
  if (defined('DB_SERVER') && defined('DB_USERNAME') && defined('DB_PASSWORD') && defined('DB_NAME') && defined('DB_PORT')) {
    $m = @new mysqli(DB_SERVER, DB_USERNAME, DB_PASSWORD, DB_NAME, DB_PORT);
    if ($m->connect_error) throw new RuntimeException('GuestPass DB connect error: ' . $m->connect_error);
    $m->set_charset('utf8mb4');
    return $m;
  }
  throw new RuntimeException('GuestPass DB config missing (config.php).');
}

/** Open mysqli connection for Guardian Assist DB (from configguardian.php) */
function gp_conn_guardian(): mysqli {
  if (function_exists('getGuardianDbConnection')) {
    $m = getGuardianDbConnection();
    if ($m instanceof mysqli) return $m;
  }
  if (defined('GA_DB_SERVER') && defined('GA_DB_USERNAME') && defined('GA_DB_PASSWORD') && defined('GA_DB_NAME') && defined('GA_DB_PORT')) {
    $m = @new mysqli(GA_DB_SERVER, GA_DB_USERNAME, GA_DB_PASSWORD, GA_DB_NAME, GA_DB_PORT);
    if ($m->connect_error) throw new RuntimeException('Guardian DB connect error: ' . $m->connect_error);
    $m->set_charset('utf8mb4');
    return $m;
  }
  throw new RuntimeException('Guardian DB config missing (configguardian.php).');
}

/* ---------------------------------------------------------------------
   Schema detection per-DB
--------------------------------------------------------------------- */

function gp_table_exists(mysqli $conn, string $table): bool {
  $db = $conn->real_escape_string($conn->query('SELECT DATABASE()')->fetch_row()[0]);
  $tableEsc = $conn->real_escape_string($table);
  $sql = "SELECT 1 FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA='{$db}' AND TABLE_NAME='{$tableEsc}' LIMIT 1";
  $res = $conn->query($sql);
  $ok = $res && $res->num_rows > 0;
  if ($res) $res->free();
  return $ok;
}

function gp_col_exists(mysqli $conn, string $table, string $col): bool {
  $db = $conn->real_escape_string($conn->query('SELECT DATABASE()')->fetch_row()[0]);
  $tableEsc = $conn->real_escape_string($table);
  $colEsc = $conn->real_escape_string($col);
  $sql = "SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA='{$db}' AND TABLE_NAME='{$tableEsc}' AND COLUMN_NAME='{$colEsc}' LIMIT 1";
  $res = $conn->query($sql);
  $ok = $res && $res->num_rows > 0;
  if ($res) $res->free();
  return $ok;
}

/**
 * Detects org table/columns for a given connection.
 * Supports:
 *  - US:  organizations(org_id, org_name, email, subscription_status, created_at)
 *  - UK:  organisations(organisation_id, organisation_name, contact_email, is_active, created_at[, contact_name, contact_phone])
 */
function gp_detect_org_schema(mysqli $conn): array {
  // prefer US naming first; fall back to UK
  $candidates = [
    [
      'table' => 'organizations',
      'id'    => 'org_id',
      'name'  => 'org_name',
      'email' => 'email',
      'status'=> 'subscription_status',
      'created'=>'created_at',
      'contact_name' => null,
      'contact_phone'=> null,
      'status_type'  => 'enum', // Active/Inactive
    ],
    [
      'table' => 'organisations',
      'id'    => 'organisation_id',
      'name'  => 'organisation_name',
      'email' => 'contact_email', // optional
      'status'=> 'is_active',     // tinyint(1)
      'created'=>'created_at',
      'contact_name' => 'contact_name',   // optional
      'contact_phone'=> 'contact_phone',  // optional
      'status_type'  => 'tinyint',
    ],
  ];

  foreach ($candidates as $sch) {
    if (!gp_table_exists($conn, $sch['table'])) continue;
    // minimally require id + name
    if (!gp_col_exists($conn, $sch['table'], $sch['id'])) continue;
    if (!gp_col_exists($conn, $sch['table'], $sch['name'])) continue;
    // email might be optional on UK schema
    if ($sch['email'] && !gp_col_exists($conn, $sch['table'], $sch['email'])) {
      // tolerate missing email; we will omit it on insert if absent
      $sch['email'] = null;
    }
    // optional contact_name/phone
    if ($sch['contact_name'] && !gp_col_exists($conn, $sch['table'], $sch['contact_name'])) $sch['contact_name'] = null;
    if ($sch['contact_phone'] && !gp_col_exists($conn, $sch['table'], $sch['contact_phone'])) $sch['contact_phone'] = null;
    // status may be absent; tolerate and set null
    if ($sch['status'] && !gp_col_exists($conn, $sch['table'], $sch['status'])) {
      $sch['status'] = null;
    }
    // created_at may be defaulted; tolerate missing and omit on insert
    if ($sch['created'] && !gp_col_exists($conn, $sch['table'], $sch['created'])) {
      $sch['created'] = null;
    }
    return $sch;
  }
  throw new RuntimeException('No organizations/organisations table detected in this database.');
}

/* ---------------------------------------------------------------------
   Generic ops using detected schema
--------------------------------------------------------------------- */

function gp_max_id(mysqli $conn, array $sch): int {
  $sql = "SELECT COALESCE(MAX(`{$sch['id']}`),0) AS m FROM `{$sch['table']}`";
  $res = $conn->query($sql);
  if (!$res) throw new RuntimeException('Failed to read max id: ' . $conn->error);
  $row = $res->fetch_assoc(); $res->free();
  return (int)($row['m'] ?? 0);
}

function gp_id_exists(mysqli $conn, array $sch, int $id): bool {
  $stmt = $conn->prepare("SELECT 1 FROM `{$sch['table']}` WHERE `{$sch['id']}` = ? LIMIT 1");
  $stmt->bind_param("i", $id);
  $stmt->execute();
  $stmt->store_result();
  $exists = $stmt->num_rows > 0;
  $stmt->free_result();
  $stmt->close();
  return $exists;
}

function gp_bump_ai(mysqli $conn, array $sch, int $id): void {
  $next = $id + 1;
  // ignore errors
  @$conn->query("ALTER TABLE `{$sch['table']}` AUTO_INCREMENT = {$next}");
}

/**
 * Insert a row using detected schema; only includes columns that exist.
 * $data can include: name, email, status, contact_name, contact_phone
 */
function gp_insert_row(mysqli $conn, array $sch, int $id, array $data): void {
  $cols = [$sch['id'] => $id, $sch['name'] => $data['name']];
  $types = 'is'; // id=int, name=string
  $vals = [$id, $data['name']];

  // optional email
  if (!empty($sch['email']) && array_key_exists('email', $data)) {
    $cols[$sch['email']] = $data['email'];
    $types .= 's';
    $vals[] = $data['email'];
  }

  // optional contact_name / contact_phone (UK)
  if (!empty($sch['contact_name']) && isset($data['contact_name'])) {
    $cols[$sch['contact_name']] = $data['contact_name'];
    $types .= 's';
    $vals[] = $data['contact_name'];
  }
  if (!empty($sch['contact_phone']) && isset($data['contact_phone'])) {
    $cols[$sch['contact_phone']] = $data['contact_phone'];
    $types .= 's';
    $vals[] = $data['contact_phone'];
  }

  // optional status
  if (!empty($sch['status']) && array_key_exists('status', $data)) {
    // if tinyint expected but given 'Active'/'Inactive', coerce to 1/0
    $val = $data['status'];
    if ($sch['status_type'] === 'tinyint') {
      $val = (strtolower((string)$val) === 'active') ? 1 : ((int)$val ? 1 : 0);
      $types .= 'i';
    } else {
      // enum path expects 'Active' or 'Inactive'
      $val = (strtolower((string)$val) === 'inactive') ? 'Inactive' : 'Active';
      $types .= 's';
    }
    $cols[$sch['status']] = $val;
    $vals[] = $val;
  }

  // created_at omitted if column missing; DB default handles it otherwise

  // Build INSERT
  $colNames = array_keys($cols);
  $ph = implode(',', array_fill(0, count($colNames), '?'));
  $sql = "INSERT INTO `{$sch['table']}` (" . implode(',', array_map(fn($c)=>"`{$c}`", $colNames)) . ") VALUES ({$ph})";

  gp_bump_ai($conn, $sch, $id);
  $stmt = $conn->prepare($sql);
  $stmt->bind_param($types, ...$vals);
  if (!$stmt->execute()) {
    $err = $stmt->error;
    $stmt->close();
    throw new RuntimeException("Insert failed ({$sch['table']} id={$id}): {$err}");
  }
  $stmt->close();
}

function gp_delete_by_id(mysqli $conn, array $sch, int $id): void {
  $stmt = $conn->prepare("DELETE FROM `{$sch['table']}` WHERE `{$sch['id']}` = ?");
  $stmt->bind_param("i", $id);
  $stmt->execute();
  $stmt->close();
}

/** Next free id available in BOTH DBs under their respective schemas */
function gp_next_free_id_both(mysqli $gp, array $gpSch, mysqli $ga, array $gaSch): int {
  $candidate = max(gp_max_id($gp, $gpSch), gp_max_id($ga, $gaSch)) + 1;
  while (true) {
    if (!gp_id_exists($gp, $gpSch, $candidate) && !gp_id_exists($ga, $gaSch, $candidate)) {
      return $candidate;
    }
    $candidate++;
  }
}

/* ---------------------------------------------------------------------
   Auth (kept: super if organisation_id === 7)
--------------------------------------------------------------------- */

$auth = $GLOBALS['AUTH'] ?? null;
if (!$auth) gp_out(['status'=>'error','message'=>'Unauthorized'], 401);
$actorOrg = (int)($auth['organisation_id'] ?? 0);
$isSuper  = ($actorOrg === 7);
if (!$isSuper) gp_out(['status'=>'error','message'=>'Forbidden'], 403);

/* ---------------------------------------------------------------------
   Input
--------------------------------------------------------------------- */

$p = gp_body_json();

// Flexible field names accepted
$orgName = trim((string)($p['org_name'] ?? $p['organisation_name'] ?? ''));
$email   = trim((string)($p['email'] ?? $p['contact_email'] ?? ''));
$status  = trim((string)($p['subscription_status'] ?? $p['is_active'] ?? 'Active'));
$contactName  = trim((string)($p['contact_name'] ?? ''));
$contactPhone = trim((string)($p['contact_phone'] ?? ''));

if ($orgName === '') gp_out(['status'=>'error','message'=>'Missing organization name'], 400);
// Allow empty email; fallback placeholder only if a schema requires an email column
if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) gp_out(['status'=>'error','message'=>'Invalid email'], 400);

/* ---------------------------------------------------------------------
   Core: detect schemas, then synchronized create
--------------------------------------------------------------------- */

try {
  $gp = gp_conn_guestpass();
  $ga = gp_conn_guardian();

  $gpSch = gp_detect_org_schema($gp);
  $gaSch = gp_detect_org_schema($ga);

  // If a side has an email column but none provided, use a placeholder
  $gpEmailNeeded = !empty($gpSch['email']) && $email === '';
  $gaEmailNeeded = !empty($gaSch['email']) && $email === '';
  $emailFinal = $email;
  if (($gpEmailNeeded || $gaEmailNeeded) && $emailFinal === '') {
    $emailFinal = 'noreply+' . time() . '@guestpass.local';
  }

  // Normalize status text for payload; per-schema coercion happens in gp_insert_row()
  if (strtolower($status) !== 'inactive' && strtolower($status) !== 'active') {
    // if a numeric was passed, treat >0 as active
    $status = ((int)$status > 0) ? 'Active' : 'Inactive';
  }

  $MAX_RETRIES = 5;
  $orgId = null;
  $attempt = 0;

  while ($attempt < $MAX_RETRIES) {
    $attempt++;
    $candidate = gp_next_free_id_both($gp, $gpSch, $ga, $gaSch);

    // Insert into GP, rollback on error
    $gp->begin_transaction();
    try {
      gp_insert_row($gp, $gpSch, $candidate, [
        'name' => $orgName,
        'email'=> $emailFinal,
        'status'=> $status,
        'contact_name' => $contactName ?: null,
        'contact_phone'=> $contactPhone ?: null,
      ]);
      $gp->commit();
    } catch (Throwable $e) {
      $gp->rollback();
      $msg = $e->getMessage();
      $dup = (stripos($msg, 'Duplicate') !== false || stripos($msg, '1062') !== false);
      gp_log("GP insert failed attempt {$attempt}: {$msg}");
      if ($dup) continue; // race — try next id
      throw $e;
    }

    // Insert into GA, rollback GP on failure
    $ga->begin_transaction();
    try {
      gp_insert_row($ga, $gaSch, $candidate, [
        'name' => $orgName,
        'email'=> $emailFinal,
        'status'=> $status,
        'contact_name' => $contactName ?: null,
        'contact_phone'=> $contactPhone ?: null,
      ]);
      $ga->commit();
      $orgId = $candidate;
      break; // success
    } catch (Throwable $e) {
      $ga->rollback();
      gp_delete_by_id($gp, $gpSch, $candidate);
      $msg = $e->getMessage();
      $dup = (stripos($msg, 'Duplicate') !== false || stripos($msg, '1062') !== false);
      gp_log("GA insert failed attempt {$attempt}: {$msg}");
      if ($dup) continue; // race — try next id
      throw $e;
    }
  }

  if ($orgId === null) gp_out(['status'=>'error','message'=>'Exhausted retries while reserving a synchronized org_id'], 500);

  // Build response using a neutral shape
  gp_out([
    'status'=>'success',
    'message'=>'Organisation created in both DBs with the same id.',
    'org'=>[
      'org_id'   => (int)$orgId,
      'name'     => $orgName,
      'email'    => $emailFinal,
      'status'   => (strtolower($status) === 'inactive' ? 'Inactive' : 'Active'),
    ],
    'gp_table' => $gpSch['table'],
    'ga_table' => $gaSch['table'],
  ], 200);

} catch (Throwable $e) {
  gp_log('Fatal: ' . $e->getMessage());
  gp_out(['status'=>'error','message'=>'Server error: '.$e->getMessage()],500);
}
