<?php
// guardianapi/org_get.php
declare(strict_types=1);
header('Content-Type: application/json; charset=UTF-8');

require_once __DIR__ . '/config.php';
$conn = getDbConnection();

// Auth (index.php sets $GLOBALS['AUTH'])
if (!isset($GLOBALS['AUTH'])) {
  http_response_code(401);
  echo json_encode(['status' => 'error', 'message' => 'Unauthorized']);
  exit;
}
$auth     = $GLOBALS['AUTH'];
$role     = (string)($auth['role'] ?? '');
$actorOrg = (int)($auth['organisation_id'] ?? -1);

// Superadmin: org 0 or 7 or SUPERADMIN_ORG_ID (if defined)
$super = ($role === 'admin') && (
  $actorOrg === 0 ||
  $actorOrg === 7 ||
  (defined('SUPERADMIN_ORG_ID') && $actorOrg === (int)SUPERADMIN_ORG_ID)
);

// Query params (all optional)
$search     = isset($_GET['search'])     ? trim((string)$_GET['search']) : '';
$is_active  = isset($_GET['is_active'])  && $_GET['is_active'] !== '' ? (int)$_GET['is_active'] : null;
$validated  = isset($_GET['validated'])  && $_GET['validated'] !== '' ? (int)$_GET['validated'] : null;
$orgIdParam = isset($_GET['organisation_id']) && $_GET['organisation_id'] !== '' ? (int)$_GET['organisation_id'] : null;
$limit      = isset($_GET['limit'])      && (int)$_GET['limit'] > 0 ? (int)$_GET['limit'] : 200;
$offset     = isset($_GET['offset'])     && (int)$_GET['offset'] >= 0 ? (int)$_GET['offset'] : 0;

// Scope: non-superadmin can only see their own organisation
$where  = [];
$params = [];
$types  = '';

if ($super) {
  if ($orgIdParam !== null) {
    $where[] = "o.organisation_id = ?";
    $types  .= "i";
    $params[] = $orgIdParam;
  }
} else {
  $where[] = "o.organisation_id = ?";
  $types  .= "i";
  $params[] = $actorOrg;
}

if ($search !== '') {
  $where[] = "(o.organisation_name LIKE CONCAT('%',?,'%')
               OR o.contact_name  LIKE CONCAT('%',?,'%')
               OR o.contact_email LIKE CONCAT('%',?,'%')
               OR o.contact_phone LIKE CONCAT('%',?,'%'))";
  $types   .= "ssss";
  $params[] = $search; $params[] = $search; $params[] = $search; $params[] = $search;
}
if ($is_active !== null) {
  $where[] = "o.is_active = ?";
  $types  .= "i";
  $params[] = $is_active;
}
if ($validated !== null) {
  $where[] = "o.validated = ?";
  $types  .= "i";
  $params[] = $validated;
}

$sql = "
  SELECT
    o.organisation_id,
    o.organisation_name,
    o.contact_name,
    o.contact_email,
    o.contact_phone,
    o.is_active,
    o.expiry,
    o.created_at,
    o.validated,
    o.validated_at,
    /* Primary admin username (oldest active admin) */
    (
      SELECT a.username
      FROM admin a
      WHERE a.organisation_id = o.organisation_id AND a.is_active = 1
      ORDER BY a.admin_id ASC
      LIMIT 1
    ) AS admin_username
  FROM organisations o
  " . (count($where) ? "WHERE " . implode(" AND ", $where) : "") . "
  ORDER BY o.organisation_id DESC
  LIMIT ? OFFSET ?
";

$types .= "ii";
$params[] = $limit;
$params[] = $offset;

$stmt = $conn->prepare($sql);
if (!$stmt) {
  http_response_code(500);
  echo json_encode(['status'=>'error','message'=>'DB error: '.$conn->error]);
  exit;
}
$stmt->bind_param($types, ...$params);
$stmt->execute();
$res = $stmt->get_result();

$out = [];
while ($r = $res->fetch_assoc()) {
  $out[] = [
    'organisation_id'   => (int)$r['organisation_id'],
    'organisation_name' => (string)$r['organisation_name'],
    'contact_name'      => $r['contact_name'],
    'contact_email'     => $r['contact_email'],
    'contact_phone'     => $r['contact_phone'],
    'is_active'         => (int)$r['is_active'],
    'expiry'            => $r['expiry'],
    'created_at'        => $r['created_at'],
    'validated'         => (int)$r['validated'],
    'validated_at'      => $r['validated_at'],
    'admin_username'    => $r['admin_username'] ?? null,
  ];
}
$stmt->close();

echo json_encode(['status'=>'success','organisations'=>$out]);
