<?php
/**
 * guardianapi/host.php
 *
 * One endpoint for:
 *   - GET  hosts   => list users (hosts) in an organisation (filters supported)
 *   - GET  houses  => list Active houses in an organisation (for linking)
 *   - POST         => create a host (user) linked to optional house_id
 *   - PATCH/POST   => update a host (username, password, real_name, is_active, house_id)
 *
 * Routing:
 *   - GET  ?resource=hosts&organisation_id=... [&active=0|1&house_id=&username=&q=]
 *   - GET  ?resource=houses&organisation_id=...
 *   - POST JSON { mode:"create", ... }
 *   - POST/PATCH JSON { mode:"update", user_id, ... }
 *
 * Uses guardianapi/config.php -> getDbConnection() (mysqli)
 */

declare(strict_types=1);

require_once __DIR__ . '/config.php';

/* ---------- CORS / Headers ---------- */
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PATCH, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ---------- Helpers (prefixed to avoid collisions) ---------- */
if (!function_exists('h3_out')) {
  function h3_out(array $arr, int $code=200): void { http_response_code($code); echo json_encode($arr); exit; }
}
if (!function_exists('h3_json')) {
  function h3_json(): array { $j=json_decode(file_get_contents('php://input'), true); return is_array($j)?$j:[]; }
}
if (!function_exists('h3_bool')) {
  function h3_bool($v): ?int {
    if ($v===null) return null;
    if (is_bool($v)) return $v?1:0;
    if (is_numeric($v)) return ((int)$v)>0?1:0;
    $s=strtolower(trim((string)$v));
    if (in_array($s,['1','true','yes','active'],true)) return 1;
    if (in_array($s,['0','false','no','inactive'],true)) return 0;
    return null;
  }
}
if (!function_exists('h3_token')) {
  function h3_token(): string { return bin2hex(random_bytes(16)); }
}

/* ---------- Routing ---------- */
$method   = strtoupper($_SERVER['REQUEST_METHOD']);
$resource = '';
if ($method === 'GET') {
  $resource = isset($_GET['resource']) ? strtolower((string)$_GET['resource']) : 'hosts';
  if ($resource === 'hosts') return h3_list_hosts();
  if ($resource === 'houses') return h3_list_houses();
  h3_out(['status'=>'error','message'=>'Unknown resource for GET. Use resource=hosts or resource=houses'], 400);
} elseif ($method === 'POST' || $method === 'PATCH') {
  $in   = h3_json();
  $mode = strtolower((string)($in['mode'] ?? ''));
  if ($mode === 'create') return h3_create_host($in);
  if ($mode === 'update') return h3_update_host($in);
  h3_out(['status'=>'error','message'=>"Missing/invalid mode. Use mode:'create' or mode:'update'"], 400);
} else {
  h3_out(['status'=>'error','message'=>'Method not allowed'], 405);
}

/* ---------- Handlers ---------- */

function h3_list_hosts(): void {
  $orgId = isset($_GET['organisation_id']) ? (int)$_GET['organisation_id'] : 0;
  if ($orgId <= 0) h3_out(['status'=>'error','message'=>'Missing or invalid organisation_id'], 400);

  $active   = isset($_GET['active']) ? h3_bool($_GET['active']) : null;
  $house_id = isset($_GET['house_id']) ? (int)$_GET['house_id'] : null;
  $username = isset($_GET['username']) ? trim((string)$_GET['username']) : '';
  $q        = isset($_GET['q']) ? trim((string)$_GET['q']) : '';

  $conn = getDbConnection();
  $conn->set_charset('utf8mb4');

  $sql = "
    SELECT u.user_id, u.organisation_id, u.real_name, u.username, u.is_active, u.house_id, u.created_at,
           h.house_number
    FROM users u
    LEFT JOIN houses h ON (u.house_id = h.id)
    WHERE u.organisation_id = ?
  ";
  $types='i'; $params=[$orgId];

  if ($active !== null) { $sql.=" AND u.is_active = ?"; $types.='i'; $params[]=$active; }
  if ($house_id !== null && $house_id>0) { $sql.=" AND u.house_id = ?"; $types.='i'; $params[]=$house_id; }
  if ($username !== '') { $sql.=" AND u.username = ?"; $types.='s'; $params[]=$username; }
  if ($q !== '') {
    $like='%'.$q.'%';
    $sql.=" AND (u.username LIKE ? OR u.real_name LIKE ?)";
    $types.='ss'; $params[]=$like; $params[]=$like;
  }

  $sql .= " ORDER BY u.created_at DESC, u.user_id DESC";

  $stmt = $conn->prepare($sql);
  if (!$stmt) h3_out(['status'=>'error','message'=>'Database error: '.$conn->error], 500);
  $stmt->bind_param($types, ...$params);
  $stmt->execute();
  $res = $stmt->get_result();

  $users=[];
  while($row=$res->fetch_assoc()){
    $row['is_active']=(int)$row['is_active'];
    $users[]=$row;
  }
  $stmt->close();
  $conn->close();

  h3_out(empty($users) ? ['status'=>'error','message'=>'No users found'] : ['status'=>'success','users'=>$users], 200);
}

function h3_list_houses(): void {
  $orgId = isset($_GET['organisation_id']) ? (int)$_GET['organisation_id'] : 0;
  if ($orgId <= 0) h3_out(['status'=>'error','message'=>'Missing or invalid organisation_id'], 400);

  $conn = getDbConnection();
  $conn->set_charset('utf8mb4');

  $sql = "
    SELECT id, organisation_id, house_number, alert_email, alert_phone, status, is_active, active_until,
           CASE WHEN status='Active' AND is_active=1 AND (active_until IS NULL OR active_until >= NOW()) THEN 1 ELSE 0 END AS effective_active
    FROM houses
    WHERE organisation_id=? AND status='Active'
    ORDER BY house_number ASC
  ";
  $stmt=$conn->prepare($sql);
  $stmt->bind_param('i',$orgId);
  $stmt->execute();
  $res=$stmt->get_result();

  $houses=[];
  while($r=$res->fetch_assoc()){
    $r['is_active']=(int)$r['is_active'];
    $r['effective_active']=(int)$r['effective_active'];
    $houses[]=$r;
  }
  $stmt->close();
  $conn->close();

  h3_out(empty($houses)?['status'=>'error','message'=>'No houses found']:['status'=>'success','houses'=>$houses], 200);
}

function h3_create_host(array $in): void {
  $organisation_id = isset($in['organisation_id']) ? (int)$in['organisation_id'] : 0;
  $real_name       = isset($in['real_name']) ? trim((string)$in['real_name']) : '';
  $username        = isset($in['username']) ? trim((string)$in['username']) : '';
  $password_plain  = isset($in['password']) ? (string)$in['password'] : '';
  $is_active       = isset($in['is_active']) ? (int)$in['is_active'] : 1;
  $house_id        = array_key_exists('house_id',$in) ? ($in['house_id']===null?null:(int)$in['house_id']) : null;

  if ($organisation_id <= 0 || $real_name === '' || $username === '' || $password_plain === '') {
    h3_out(['status'=>'error','message'=>'Missing organisation_id, real_name, username or password'], 400);
  }

  $conn = getDbConnection();
  $conn->set_charset('utf8mb4');
  $conn->begin_transaction();

  try {
    /* ---- GLOBAL username uniqueness (table-wide) ---- */
    $stmt = $conn->prepare("SELECT user_id FROM users WHERE username=? LIMIT 1");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $dupe = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    if ($dupe) { $conn->rollback(); h3_out(['status'=>'error','message'=>'Username already exists'], 409); }

    // Validate house (if provided): must be same org and not soft-deleted
    if ($house_id !== null) {
      $stmt = $conn->prepare("SELECT id FROM houses WHERE id=? AND organisation_id=? AND status='Active' LIMIT 1");
      $stmt->bind_param("ii", $house_id, $organisation_id);
      $stmt->execute();
      $okHouse = $stmt->get_result()->fetch_assoc();
      $stmt->close();
      if (!$okHouse) { $conn->rollback(); h3_out(['status'=>'error','message'=>'Invalid house_id for this organisation'], 400); }
    }

    $pwd_hash = password_hash($password_plain, PASSWORD_DEFAULT);
    $token    = h3_token();

    $stmt = $conn->prepare("INSERT INTO users (organisation_id, real_name, username, password, token, is_active, house_id, created_at) VALUES (?,?,?,?,?,?,?, NOW())");
    $stmt->bind_param("issssii", $organisation_id, $real_name, $username, $pwd_hash, $token, $is_active, $house_id);
    $stmt->execute();
    $uid = (int)$stmt->insert_id;
    $stmt->close();

    $conn->commit();
    h3_out([
      'status'=>'success',
      'message'=>'Host account created',
      'data'=>[
        'user_id'=>$uid,
        'organisation_id'=>$organisation_id,
        'real_name'=>$real_name,
        'username'=>$username,
        'is_active'=>$is_active,
        'house_id'=>$house_id,
        'token'=>$token
      ]
    ], 201);

  } catch (Throwable $e) {
    $conn->rollback();
    // normalize duplicate-key errors if a DB unique index exists
    $msg = $e->getMessage();
    if (strpos($msg, 'Duplicate') !== false || strpos($msg, '1062') !== false) {
      h3_out(['status'=>'error','message'=>'Username already exists'], 409);
    }
    h3_out(['status'=>'error','message'=>'Create failed: '.$msg], 500);
  }
}

function h3_update_host(array $in): void {
  $user_id = isset($in['user_id']) ? (int)$in['user_id'] : 0;
  if ($user_id <= 0) h3_out(['status'=>'error','message'=>'user_id required'], 400);

  $conn = getDbConnection();
  $conn->set_charset('utf8mb4');
  $conn->begin_transaction();

  try {
    // Load current user (to get org for house validation)
    $stmt = $conn->prepare("SELECT user_id, organisation_id, real_name, username, is_active, house_id FROM users WHERE user_id=? LIMIT 1");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    if (!$row) { $conn->rollback(); h3_out(['status'=>'error','message'=>'User not found'], 404); }

    $organisation_id = (int)$row['organisation_id'];

    // Proposed changes
    $real_name = array_key_exists('real_name',$in) ? trim((string)$in['real_name']) : null;
    $username  = array_key_exists('username',$in) ? trim((string)$in['username']) : null;
    $password  = array_key_exists('password',$in) ? (string)$in['password'] : null;
    $is_active = array_key_exists('is_active',$in) ? (int)$in['is_active'] : null;
    $house_id  = array_key_exists('house_id',$in) ? ($in['house_id']===null?null:(int)$in['house_id']) : '__NOCHANGE__';

    /* ---- GLOBAL username uniqueness on change ---- */
    if ($username !== null && $username !== '' && $username !== $row['username']) {
      $stmt = $conn->prepare("SELECT user_id FROM users WHERE username=? AND user_id<>? LIMIT 1");
      $stmt->bind_param("si", $username, $user_id);
      $stmt->execute();
      $dupe = $stmt->get_result()->fetch_assoc();
      $stmt->close();
      if ($dupe) { $conn->rollback(); h3_out(['status'=>'error','message'=>'Username already exists'], 409); }
    }

    // Validate new house link (if changing and not null). House must belong to same org and not be Deleted.
    if ($house_id !== '__NOCHANGE__' && $house_id !== null) {
      $stmt = $conn->prepare("SELECT id FROM houses WHERE id=? AND organisation_id=? AND status='Active' LIMIT 1");
      $stmt->bind_param("ii", $house_id, $organisation_id);
      $stmt->execute();
      $ok = $stmt->get_result()->fetch_assoc();
      $stmt->close();
      if (!$ok) { $conn->rollback(); h3_out(['status'=>'error','message'=>'Invalid house_id for this organisation'], 400); }
    }

    // Build dynamic update
    $set=[]; $types=''; $vals=[];

    if ($real_name !== null) { $set[]="real_name=?"; $types.='s'; $vals[]=$real_name; }
    if ($username  !== null && $username !== '') { $set[]="username=?"; $types.='s'; $vals[]=$username; }
    if ($password  !== null && $password !== '') { $set[]="password=?"; $types.='s'; $vals[]=password_hash($password, PASSWORD_DEFAULT); }
    if ($is_active !== null) { $set[]="is_active=?"; $types.='i'; $vals[]=$is_active; }
    if ($house_id  !== '__NOCHANGE__') { $set[]="house_id=?"; $types.='i'; $vals[]=$house_id; }

    if (empty($set)) { $conn->rollback(); h3_out(['status'=>'error','message'=>'No fields to update'], 400); }

    $sql = "UPDATE users SET ".implode(',', $set)." WHERE user_id=?";
    $types.='i'; $vals[]=$user_id;

    $stmt=$conn->prepare($sql);
    $stmt->bind_param($types, ...$vals);
    $stmt->execute();
    $stmt->close();

    // Read back
    $stmt = $conn->prepare("SELECT user_id, organisation_id, real_name, username, is_active, house_id, created_at FROM users WHERE user_id=? LIMIT 1");
    $stmt->bind_param("i",$user_id);
    $stmt->execute();
    $data = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    $conn->commit();
    h3_out(['status'=>'success','message'=>'Host updated','data'=>$data], 200);

  } catch (Throwable $e) {
    $conn->rollback();
    $msg = $e->getMessage();
    if (strpos($msg, 'Duplicate') !== false || strpos($msg, '1062') !== false) {
      h3_out(['status'=>'error','message'=>'Username already exists'], 409);
    }
    h3_out(['status'=>'error','message'=>'Update failed: '.$msg], 500);
  }
}
